<?php
/* --------------------------------------------------------------
 AdminMenuServiceProvider.php 2020-04-23
 Gambio GmbH
 http://www.gambio.de
 Copyright (c) 31 Gambio GmbH
 Released under the GNU General Public License (Version 2)
 [http://www.gnu.org/licenses/gpl-2.0.html]
 --------------------------------------------------------------
 */

declare(strict_types=1);

namespace Gambio\Admin\Layout\Menu;

use Doctrine\DBAL\Connection;
use Gambio\Core\Application\ValueObjects\Path;
use Gambio\Core\Application\ValueObjects\Server;
use Gambio\Core\Application\ValueObjects\Url;
use Gambio\Core\Application\ValueObjects\UserPreferences;
use Gambio\Core\Configuration\ConfigurationService;
use Gambio\Admin\Layout\Menu\CommandHandlers\ClearMenuCacheHandler;
use Gambio\Admin\Layout\Menu\Commands\ClearMenuCache;
use Gambio\Admin\Layout\Menu\Compatibility\GXModulesJsonUtility;
use Gambio\Admin\Layout\Menu\EventListeners\GxModules\Json\JsonMenuFinder;
use Gambio\Admin\Layout\Menu\EventListeners\GxModules\Json\JsonMenuListener;
use Gambio\Admin\Layout\Menu\EventListeners\GxModules\ModulesFinder;
use Gambio\Admin\Layout\Menu\EventListeners\GxModules\Xml\XmlMenuFinder;
use Gambio\Admin\Layout\Menu\EventListeners\GxModules\Xml\XmlMenuListener;
use Gambio\Admin\Layout\Menu\Events\CoreMenuDataCollected;
use Gambio\Admin\Layout\Menu\Factories\CacheMenuFactory;
use Gambio\Admin\Layout\Menu\Factories\Helper\Utility;
use Gambio\Admin\Layout\Menu\Factories\Helper\Verifier;
use Gambio\Admin\Layout\Menu\Factories\PostCacheMenuFactory;
use Gambio\Admin\Layout\Menu\Filter\FilterFactory;
use Gambio\Admin\Layout\Menu\Models\Cached\MenuSettings;
use Gambio\Admin\Layout\Menu\Repositories\Cache\AdminMenuCacheRepository;
use Gambio\Admin\Layout\Menu\Repositories\Cache\Implementation\MenuCache;
use Gambio\Admin\Layout\Menu\Repositories\Cache\Implementation\MenuMapper;
use Gambio\Admin\Layout\Menu\Repositories\PostCache\FavouritesHashListRepository;
use Gambio\Admin\Layout\Menu\Repositories\PostCache\Implementation\FavouritesHashListDbRepository;
use Gambio\Admin\Layout\Menu\Repositories\PostCache\Implementation\MenuProcessor;
use Gambio\Core\Application\Application;
use Gambio\Core\Application\ServiceProviders\Abstraction\AbstractBootableServiceProvider;
use Gambio\Core\Cache\CacheFactory;
use Gambio\Core\Command\Interfaces\CommandHandlerProvider;
use Gambio\Core\Event\EventListenerProvider;
use Gambio\Core\Language\LanguageService;
use Gambio\Core\Language\TextManager;
use Gambio\Core\Logging\LoggerBuilder;
use Gambio\Core\Permission\PermissionRepository;
use Psr\EventDispatcher\EventDispatcherInterface;

/**
 * Class AdminMenuServiceProvider
 * @package Gambio\Admin\Layout\Menu
 *
 * @property-read Application $container
 */
class AdminMenuServiceProvider extends AbstractBootableServiceProvider
{
    /**
     * @return array
     */
    public function provides(): array
    {
        return [
            AdminMenuService::class,
            GXModulesJsonUtility::class,
            
            ClearMenuCacheHandler::class
        ];
    }
    
    
    /**
     * @inheritDoc
     */
    public function boot(): void
    {
        /** @var EventListenerProvider $eventListener */
        /** @var CommandHandlerProvider $commandHandler */
        $eventListener  = $this->application->get(EventListenerProvider::class);
        $commandHandler = $this->application->get(CommandHandlerProvider::class);
        
        $eventListener->attachListener(CoreMenuDataCollected::class, XmlMenuListener::class);
        $eventListener->attachListener(CoreMenuDataCollected::class, JsonMenuListener::class);
        
        $commandHandler->attachHandler(ClearMenuCache::class, ClearMenuCacheHandler::class);
    }
    
    
    /**
     * @inheritDoc
     */
    public function register(): void
    {
        $this->registerService();
        $this->registerCacheRepository();
        $this->registerMenuProcessor();
        $this->registerEventListeners();
        $this->registerCommandHandlers();
        $this->registerUtilityComponents();
    }
    
    
    private function registerService(): void
    {
        $this->application->share(AdminMenuService::class)
            ->addArgument(AdminMenuCacheRepository::class)
            ->addArgument(MenuProcessor::class);
    }
    
    
    private function registerCacheRepository(): void
    {
        $this->application->share(AdminMenuCacheRepository::class,
                                  Repositories\Cache\Implementation\AdminMenuCacheRepository::class)
            ->addArgument(MenuCache::class)
            ->addArgument(MenuMapper::class)
            ->addArgument(FilterFactory::class)
            ->addArgument(LanguageService::class)
            ->addArgument(EventDispatcherInterface::class)
            ->addArgument(LoggerBuilder::class);
        
        $this->application->share(MenuCache::class)
            ->addArgument(CacheFactory::class)
            ->addArgument(UserPreferences::class)
            ->addArgument(LoggerBuilder::class);
        
        $this->application->share(MenuMapper::class)->addArgument(CacheMenuFactory::class);
        $this->application->share(CacheMenuFactory::class)
            ->addArgument(TextManager::class)
            ->addArgument(UserPreferences::class);
        
        $this->application->share(FilterFactory::class, Factories\FilterFactory::class)
            ->addArgument(ConfigurationService::class)
            ->addArgument(Path::class);
    }
    
    
    private function registerMenuProcessor(): void
    {
        $this->application->share(MenuProcessor::class)
            ->addArgument(FavouritesHashListRepository::class)
            ->addArgument(PostCacheMenuFactory::class);
        
        $this->application->share(FavouritesHashListRepository::class,
                                  FavouritesHashListDbRepository::class)
            ->addArgument(Connection::class)
            ->addArgument(UserPreferences::class);
        
        $this->application->share(PostCacheMenuFactory::class)
            ->addArgument(MenuSettings::class)
            ->addArgument(Verifier::class)
            ->addArgument(EventDispatcherInterface::class);
        
        $this->application->share(MenuSettings::class)->addArgument(Url::class)->addArgument(Server::class);
        
        $this->application->share(Verifier::class)
            ->addArgument(PermissionRepository::class)
            ->addArgument(Utility::class)
            ->addArgument(UserPreferences::class);
        
        $this->application->share(Utility::class)->addArgument(MenuSettings::class);
    }
    
    
    private function registerEventListeners(): void
    {
        $this->application->share(XmlMenuListener::class)
            ->addArgument(XmlMenuFinder::class)
            ->addArgument(CacheMenuFactory::class);
        $this->application->share(XmlMenuFinder::class)->addArgument(ModulesFinder::class);
        
        $this->application->share(JsonMenuListener::class)
            ->addArgument(JsonMenuFinder::class)
            ->addArgument(MenuMapper::class);
        $this->application->share(JsonMenuFinder::class)->addArgument(ModulesFinder::class);
        
        $this->application->share(ModulesFinder::class)->addArgument(Path::class);
    }
    
    
    private function registerCommandHandlers(): void
    {
        $this->application->share(ClearMenuCacheHandler::class)->addArgument(AdminMenuService::class);
    }
    
    
    private function registerUtilityComponents(): void
    {
        $this->application->share(GXModulesJsonUtility::class)
            ->addArgument(JsonMenuFinder::class)
            ->addArgument(MenuMapper::class);
    }
}